/**
 * SPDX license identifier: MPL-2.0
 *
 * Copyright (C) 2017, ADIT GmbH
 *
 * This file is part of GENIVI Project AudioManager.
 *
 * Contributions are licensed to the GENIVI Alliance under one or more
 * Contribution License Agreements.
 *
 * \copyright
 * This Source Code Form is subject to the terms of the
 * Mozilla Public License, v. 2.0. If a  copy of the MPL was not distributed with
 * this file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * \author Jens Lorenz, jlorenz@de.adit-jv.com ADIT 2017
 * \author Mattia Guerra, mguerra@de.adit-jv.com ADIT 2017
 *
 * \file CAmLoggerStdOut.cpp
 * For further information see http://www.genivi.org/.
 *
 */

#include <algorithm>
#include <iostream>
#include <sstream>
#include <mutex>
#include <iomanip>
#include "CAmLogWrapper.h"
#include "CAmLoggerStdOut.h"
#include "CAmTimeUtility.h"

using namespace std;

namespace am
{

#define PADDING_WIDTH 4
#define PADDING_WAY setw(PADDING_WIDTH)
#define PADDING_ALIGN left

#define CC_BLACK    "\x1b[0;30m"
#define CC_RED      "\x1b[0;31m"
#define CC_GREEN    "\x1b[0;32m"
#define CC_YELLOW   "\x1b[0;33m"
#define CC_BLUE     "\x1b[0;34m"
#define CC_MAGENTA  "\x1b[0;35m"
#define CC_CYAN     "\x1b[0;36m"
#define CC_WHITE    "\x1b[0;37m"
#define CC_RESET    "\x1b[0;39m"


pthread_mutex_t gStdOutMtx = PTHREAD_MUTEX_INITIALIZER;

CAmLogContextStdOut::CAmLogContextStdOut(const char* id, const am_LogLevel_e level) :
    mId(id), mLogLevel(level)
{
}

void CAmLogContextStdOut::append(const int8_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const uint8_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const int16_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const uint16_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const int32_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const uint32_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const uint64_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const int64_t value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const bool value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const char * value)
{
    appendStdOut(value);
}

void CAmLogContextStdOut::append(const vector<uint8_t> & data)
{
    mBuffer << data.data() << " ";
}

template<class T> void CAmLogContextStdOut::appendStdOut(T value)
{
    mBuffer << value << " ";
}

bool CAmLogContextStdOut::configure(const am_LogLevel_e loglevel)
{
    if (loglevel > mLogLevel)
        return false;

    pthread_mutex_lock(&gStdOutMtx);

    // Print time and context
    cout << CAmTimeUtility::now();
    cout << CC_GREEN << "[" << PADDING_WAY << PADDING_ALIGN << string(mId, 0, PADDING_WIDTH) << "] ";

    // and late a colorful payload ...
    switch (loglevel)
    {
        case LL_ERROR:
            cout << CC_RED;
            break;
        case LL_WARN:
            cout << CC_YELLOW;
            break;
        default:
            cout << CC_RESET;
            break;
    }

    return true;
}

bool CAmLogContextStdOut::checkLogLevel(const am_LogLevel_e logLevel)
{
    return logLevel <= mLogLevel;
}

void CAmLogContextStdOut::send()
{
    // print now the payload
    // NOTE: The color is set in the configure function
    cout << mBuffer.str().c_str() << CC_RESET << endl;
    mBuffer.str("");
    mBuffer.clear();
    pthread_mutex_unlock(&gStdOutMtx);
}



CAmLoggerStdOut::CAmLoggerStdOut(const bool debugOn, const bool onlyError) : mDebugOn(debugOn), mStandardLogLevel(LL_INFO)
{
    if (onlyError)
    {
        mStandardLogLevel = LL_ERROR;
    }
    if (!mDebugOn)
        print("Running without Logging support");
}

CAmLoggerStdOut::~CAmLoggerStdOut()
{
    unregisterApp();
}

void CAmLoggerStdOut::unregisterApp()
{
    for (auto&& context : mCtxTable)
        unregisterContext(context.first);
}

void CAmLoggerStdOut::registerApp(const char *appid, const char * description)

{
    print("Register Application " + string(appid, PADDING_WIDTH) + ", " + string(description));
    registerContext(DEFAULT_CONTEXT, DEFAULT_DESCRIPTION);
}

IAmLogContext& CAmLoggerStdOut::registerContext(const char * contextid, const char * description)
{
    return registerContext(contextid, description, mStandardLogLevel, (mDebugOn) ? LS_ON : LS_OFF);
}

IAmLogContext& CAmLoggerStdOut::registerContext(const char * contextid, const char * description,
        const am_LogLevel_e level, const am_LogStatus_e status)
{
    auto&& entry = mCtxTable.find(contextid);
    if (entry == mCtxTable.end())
    {
        std::pair<std::map<const char*, CAmLogContextStdOut*>::iterator, bool> result = mCtxTable.insert(std::make_pair(contextid, new CAmLogContextStdOut(contextid, level)));
        entry = result.first;
    }

    if (mDebugOn)
        print("Registering Context " + string(contextid, PADDING_WIDTH) + ", " + description);

    return *entry->second;
}

IAmLogContext& CAmLoggerStdOut::importContext(const char* contextid)
{
    return *(mCtxTable[contextid ? contextid : DEFAULT_CONTEXT]);
}

void CAmLoggerStdOut::unregisterContext(const char* contextid)
{
    delete mCtxTable[contextid];
    mCtxTable.erase(contextid);
    print("Context " + string(contextid, PADDING_WIDTH) + "unregistered");
}

void CAmLoggerStdOut::print(string str)
{
    cout << CAmTimeUtility::now() << CC_BLUE << "[" << PADDING_WAY << PADDING_ALIGN << "LOG" << "] " << CC_RESET << str << endl;
}
}
